<?php
declare(strict_types=1);

namespace App\Services;

final class UploadService
{
  private array $allowed = [
    'image/jpeg' => 'jpg',
    'image/png'  => 'png',
    'image/webp' => 'webp',
  ];

  public function saveImage(array $file, string $type): ?string
  {
    if (($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) return null;

    $maxMb = (int)($_ENV['MAX_UPLOAD_MB'] ?? 6);
    if (($file['size'] ?? 0) > $maxMb * 1024 * 1024) return null;

    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    if (!isset($this->allowed[$mime])) return null;

    $ext = $this->allowed[$mime];
    $baseDir = __DIR__ . '/../../storage/uploads/';
    $sub = $type === 'logo' ? 'logos' : 'gallery';
    $dir = $baseDir . $sub;
    if (!is_dir($dir)) @mkdir($dir, 0775, true);

    $name = bin2hex(random_bytes(16)) . '.' . $ext;
    $dest = $dir . '/' . $name;

    $ok = $this->resizeAndSave($file['tmp_name'], $dest, $mime, $type === 'logo' ? 512 : 1400);
    if (!$ok) return null;

    return 'storage/uploads/' . $sub . '/' . $name;
  }

  private function resizeAndSave(string $src, string $dest, string $mime, int $maxWidth): bool
  {
    if (extension_loaded('imagick')) {
      try {
        $img = new \Imagick($src);
        $img->setImageOrientation(\Imagick::ORIENTATION_TOPLEFT);
        $img->stripImage();
        $img->thumbnailImage($maxWidth, 0);
        $img->setImageCompressionQuality(85);
        $img->writeImage($dest);
        $img->clear();
        return true;
      } catch (\Throwable $e) { return false; }
    }

    $create = match($mime) {
      'image/jpeg' => 'imagecreatefromjpeg',
      'image/png' => 'imagecreatefrompng',
      'image/webp' => 'imagecreatefromwebp',
      default => null
    };
    $save = match($mime) {
      'image/jpeg' => fn($im) => imagejpeg($im, $dest, 85),
      'image/png' => fn($im) => imagepng($im, $dest, 6),
      'image/webp' => fn($im) => imagewebp($im, $dest, 85),
      default => null
    };
    if (!$create || !$save || !function_exists($create)) return false;

    $srcIm = @$create($src);
    if (!$srcIm) return false;

    $w = imagesx($srcIm); $h = imagesy($srcIm);
    if ($w <= 0 || $h <= 0) return false;

    $newW = min($maxWidth, $w);
    $newH = (int)round($h * ($newW / $w));

    $dstIm = imagecreatetruecolor($newW, $newH);
    imagealphablending($dstIm, true);
    imagesavealpha($dstIm, true);

    imagecopyresampled($dstIm, $srcIm, 0,0,0,0, $newW,$newH, $w,$h);

    $save($dstIm);
    imagedestroy($srcIm);
    imagedestroy($dstIm);
    return true;
  }
}
