<?php
declare(strict_types=1);

namespace App\Services;

use App\Core\DB;
use App\Core\Helpers;

final class SearchService
{
  public function search(array $q): array
  {
    $term = Helpers::norm((string)($q['q'] ?? ''));
    $city = trim((string)($q['city'] ?? ($_ENV['DEFAULT_CITY'] ?? '')));
    $district = trim((string)($q['district'] ?? ''));
    $type = in_array(($q['type'] ?? ''), ['company','professional'], true) ? $q['type'] : null;
    $categorySlug = trim((string)($q['category'] ?? ''));
    $radiusKm = (float)($q['radius'] ?? 0);
    $lat = isset($q['lat']) ? (float)$q['lat'] : null;
    $lng = isset($q['lng']) ? (float)$q['lng'] : null;
    $openNow = !empty($q['open_now']);
    $minRating = (int)($q['min_rating'] ?? 0);
    $sort = $q['sort'] ?? 'relevance';

    $pdo = DB::pdo();

    $params = [];
    $joins = [];
    $wheres = ["a.city = ?"];
    $params[] = $city;

    if ($district !== '') { $wheres[] = "a.district = ?"; $params[] = $district; }
    if ($type) { $wheres[] = "l.type = ?"; $params[] = $type; }
    if ($categorySlug !== '') {
      $joins[] = "JOIN listing_categories lc ON lc.listing_id=l.id";
      $joins[] = "JOIN categories c ON c.id=lc.category_id";
      $wheres[] = "c.slug = ?";
      $params[] = $categorySlug;
    }

    $termSql = '';
    if ($term !== '') {
      // Simples ranking: título pesa mais que bio; serviços pesa menos
      $joins[] = "LEFT JOIN services s ON s.listing_id=l.id AND s.active=1";
      $termLike = '%' . $term . '%';

      $termSql = """
        (
          (CASE WHEN l.title LIKE ? THEN 3 ELSE 0 END) +
          (CASE WHEN l.bio   LIKE ? THEN 1 ELSE 0 END) +
          (CASE WHEN s.name  LIKE ? THEN 1 ELSE 0 END) +
          (CASE WHEN s.description LIKE ? THEN 0.5 ELSE 0 END)
        )
      """;

      // FULLTEXT fallback (quando possível) para melhorar (não quebra se não usar)
      $termSql = "($termSql + (CASE WHEN MATCH(l.title,l.bio) AGAINST (? IN NATURAL LANGUAGE MODE) THEN 1 ELSE 0 END))";

      $params = array_merge($params, [$termLike,$termLike,$termLike,$termLike, $term]);
      $wheres[] = "$termSql > 0";
    } else {
      $termSql = "0";
    }

    // rating
    $joins[] = "LEFT JOIN (
      SELECT listing_id, AVG(rating) avg_rating, COUNT(*) cnt
      FROM reviews
      WHERE status='approved'
      GROUP BY listing_id
    ) r ON r.listing_id=l.id";

    if ($minRating > 0) {
      $wheres[] = "COALESCE(r.avg_rating,0) >= ?";
      $params[] = $minRating;
    }

    // open now
    if ($openNow) {
      $weekday = (int)date('w');
      $now = date('H:i:s');
      $joins[] = "LEFT JOIN listing_hours h ON h.listing_id=l.id AND h.weekday=?";
      $params[] = $weekday;
      $wheres[] = "(h.is_closed=0 AND h.open_time <= ? AND h.close_time >= ?)";
      $params[] = $now; $params[] = $now;
    }

    // distance
    $distanceSql = "NULL";
    if ($lat !== null && $lng !== null) {
      $distanceSql = "(6371 * acos( cos(radians(?)) * cos(radians(a.lat)) * cos(radians(a.lng) - radians(?)) + sin(radians(?)) * sin(radians(a.lat)) ))";
      $params = array_merge([$lat, $lng, $lat], $params); // prepend for select
      if ($radiusKm > 0) {
        $wheres[] = "a.lat IS NOT NULL AND a.lng IS NOT NULL AND $distanceSql <= ?";
        $params[] = $radiusKm;
      } else {
        $wheres[] = "a.lat IS NOT NULL AND a.lng IS NOT NULL";
      }
    }

    $sql = "SELECT
      l.id,l.type,l.title,l.slug,l.plan,l.status,
      a.district,a.city,a.state,a.lat,a.lng,
      COALESCE(r.avg_rating,0) AS avg_rating,
      COALESCE(r.cnt,0) AS reviews_count,
      $termSql AS relevance,
      $distanceSql AS distance_km
    FROM listings l
    JOIN addresses a ON a.listing_id=l.id
    " . implode("\n", array_unique($joins)) . "
    WHERE l.status='active' AND " . implode(' AND ', $wheres) . "
    GROUP BY l.id
    ";

    $order = match($sort) {
      'near' => "ORDER BY distance_km ASC",
      'rating' => "ORDER BY avg_rating DESC, reviews_count DESC",
      'recent' => "ORDER BY l.created_at DESC",
      default => "ORDER BY (CASE WHEN l.plan='premium' AND (l.premium_until IS NULL OR l.premium_until>=NOW()) THEN 1 ELSE 0 END) DESC, relevance DESC, avg_rating DESC"
    };
    $sql .= "\n" . $order . "\nLIMIT 80";

    $st = $pdo->prepare($sql);
    $st->execute($params);
    $rows = $st->fetchAll();

    return $rows ?: [];
  }
}
