<?php
declare(strict_types=1);

namespace App\Models;

use PDO;

final class Listing extends Model
{
  public static function findBySlug(string $slug): ?array
  {
    $pdo = self::pdo();
    $st = $pdo->prepare("SELECT l.*, a.street,a.number,a.district,a.city,a.state,a.zipcode,a.lat,a.lng,
      COALESCE(r.avg_rating,0) avg_rating, COALESCE(r.cnt,0) reviews_count
      FROM listings l
      JOIN addresses a ON a.listing_id=l.id
      LEFT JOIN (SELECT listing_id, AVG(rating) avg_rating, COUNT(*) cnt FROM reviews WHERE status='approved' GROUP BY listing_id) r
        ON r.listing_id=l.id
      WHERE l.slug=? AND l.status IN ('active','pending') LIMIT 1");
    $st->execute([$slug]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  }

  public static function findByUser(int $userId): ?array
  {
    $st = self::pdo()->prepare("SELECT l.*, a.street,a.number,a.district,a.city,a.state,a.zipcode,a.lat,a.lng
      FROM listings l
      JOIN addresses a ON a.listing_id=l.id
      WHERE l.user_id=? LIMIT 1");
    $st->execute([$userId]);
    $row=$st->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  }

  public static function createForUser(int $userId, string $type, string $title, string $slug, string $city, string $state): int
  {
    $pdo=self::pdo();
    $st=$pdo->prepare("INSERT INTO listings (user_id,type,title,slug,status) VALUES (?,?,?,?, 'pending')");
    $st->execute([$userId,$type,$title,$slug]);
    $id=(int)$pdo->lastInsertId();

    $st2=$pdo->prepare("INSERT INTO addresses (listing_id,city,state) VALUES (?,?,?)");
    $st2->execute([$id,$city,$state]);
    return $id;
  }

  public static function updateProfile(int $listingId, array $data): void
  {
    $pdo=self::pdo();
    $st=$pdo->prepare("UPDATE listings SET title=?, bio=?, phone=?, whatsapp=?, email_public=?, website=?, budget_button_enabled=? WHERE id=?");
    $st->execute([
      $data['title'], $data['bio'], $data['phone'], $data['whatsapp'],
      $data['email_public'], $data['website'], (int)$data['budget_button_enabled'], $listingId
    ]);

    $st2=$pdo->prepare("UPDATE addresses SET street=?, number=?, district=?, city=?, state=?, zipcode=?, lat=?, lng=? WHERE listing_id=?");
    $st2->execute([
      $data['street'], $data['number'], $data['district'], $data['city'], $data['state'],
      $data['zipcode'], $data['lat'], $data['lng'], $listingId
    ]);

    // categories sync
    $pdo->prepare("DELETE FROM listing_categories WHERE listing_id=?")->execute([$listingId]);
    if (!empty($data['category_ids']) && is_array($data['category_ids'])) {
      $ins=$pdo->prepare("INSERT INTO listing_categories (listing_id,category_id) VALUES (?,?)");
      foreach ($data['category_ids'] as $cid) $ins->execute([$listingId,(int)$cid]);
    }
  }

  public static function categories(int $listingId): array
  {
    $st=self::pdo()->prepare("SELECT c.* FROM categories c JOIN listing_categories lc ON lc.category_id=c.id WHERE lc.listing_id=? ORDER BY c.name");
    $st->execute([$listingId]);
    return $st->fetchAll() ?: [];
  }

  public static function services(int $listingId): array
  {
    $st=self::pdo()->prepare("SELECT * FROM services WHERE listing_id=? ORDER BY id DESC");
    $st->execute([$listingId]);
    return $st->fetchAll() ?: [];
  }

  public static function media(int $listingId): array
  {
    $st=self::pdo()->prepare("SELECT * FROM media WHERE listing_id=? ORDER BY id DESC");
    $st->execute([$listingId]);
    return $st->fetchAll() ?: [];
  }

  public static function hours(int $listingId): array
  {
    $st=self::pdo()->prepare("SELECT * FROM listing_hours WHERE listing_id=? ORDER BY weekday ASC");
    $st->execute([$listingId]);
    $rows=$st->fetchAll() ?: [];
    $by=[];
    foreach ($rows as $r) $by[(int)$r['weekday']]=$r;
    return $by;
  }

  public static function setLatLng(int $listingId, float $lat, float $lng): void
  {
    $st=self::pdo()->prepare("UPDATE addresses SET lat=?, lng=? WHERE listing_id=?");
    $st->execute([$lat,$lng,$listingId]);
  }

  public static function setStatus(int $listingId, string $status): void
  {
    $st=self::pdo()->prepare("UPDATE listings SET status=? WHERE id=?");
    $st->execute([$status,$listingId]);
  }

  public static function setPlan(int $listingId, string $plan, ?string $premiumUntil=null): void
  {
    $st=self::pdo()->prepare("UPDATE listings SET plan=?, premium_until=? WHERE id=?");
    $st->execute([$plan,$premiumUntil,$listingId]);
  }
}
