<?php
declare(strict_types=1);

namespace App\Core;

final class RateLimiter
{
  public static function hit(string $key, int $limit, int $windowSeconds): bool
  {
    $driver = $_ENV['RATE_LIMIT_DRIVER'] ?? 'file';
    if ($driver === 'redis' && extension_loaded('redis')) {
      try {
        $r = new \Redis();
        $r->connect($_ENV['REDIS_HOST'] ?? '127.0.0.1', (int)($_ENV['REDIS_PORT'] ?? 6379), 1.0);
        $redisKey = "rl:" . $key;
        $count = (int)$r->incr($redisKey);
        if ($count === 1) $r->expire($redisKey, $windowSeconds);
        return $count <= $limit;
      } catch (\Throwable $e) { /* fallback */ }
    }

    $dir = __DIR__ . '/../../storage/cache';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);
    $file = $dir . '/rl_' . hash('sha256', $key) . '.json';

    $now = time();
    $data = ['start' => $now, 'count' => 0];

    if (is_file($file)) {
      $raw = file_get_contents($file);
      $parsed = json_decode($raw ?: '[]', true);
      if (is_array($parsed) && isset($parsed['start'], $parsed['count'])) $data = $parsed;
    }

    if (($now - (int)$data['start']) >= $windowSeconds) $data = ['start' => $now, 'count' => 0];

    $data['count'] = (int)$data['count'] + 1;
    file_put_contents($file, json_encode($data), LOCK_EX);

    return $data['count'] <= $limit;
  }
}
