<?php
declare(strict_types=1);

namespace App\Core;

use App\Middlewares\CsrfMiddleware;

final class App
{
  private Router $router;

  public function __construct()
  {
    $this->router = new Router();
    $this->registerRoutes();
  }

  private function registerRoutes(): void
  {
    $this->router->get('/', 'HomeController@index');
    $this->router->get('/buscar', 'SearchController@index');
    $this->router->get('/categoria/{slug}', 'CategoryController@show');
    $this->router->get('/perfil/{slug}', 'ListingController@show');
    $this->router->get('/sitemap.xml', 'SitemapController@index');

    $this->router->get('/entrar', 'AuthController@loginForm');
    $this->router->post('/entrar', 'AuthController@login', [CsrfMiddleware::class]);
    $this->router->get('/sair', 'AuthController@logout');
    $this->router->get('/registrar', 'AuthController@registerForm');
    $this->router->post('/registrar', 'AuthController@register', [CsrfMiddleware::class]);
    $this->router->get('/verificar-email', 'AuthController@verifyEmail');
    $this->router->get('/esqueci-senha', 'AuthController@forgotForm');
    $this->router->post('/esqueci-senha', 'AuthController@forgot', [CsrfMiddleware::class]);
    $this->router->get('/resetar-senha', 'AuthController@resetForm');
    $this->router->post('/resetar-senha', 'AuthController@reset', [CsrfMiddleware::class]);

    $this->router->post('/favoritar', 'FavoriteController@toggle', [CsrfMiddleware::class]);
    $this->router->post('/avaliar', 'ReviewController@store', [CsrfMiddleware::class]);
    $this->router->post('/responder-avaliacao', 'ReviewController@reply', [CsrfMiddleware::class]);
    $this->router->post('/contato', 'LeadController@store', [CsrfMiddleware::class]);
    $this->router->post('/denunciar', 'ReportController@store', [CsrfMiddleware::class]);

    $this->router->get('/painel', 'DashboardController@index');
    $this->router->get('/painel/perfil', 'DashboardController@editProfile');
    $this->router->post('/painel/perfil', 'DashboardController@updateProfile', [CsrfMiddleware::class]);
    $this->router->post('/painel/endereco-geocode', 'DashboardController@geocodeAddress', [CsrfMiddleware::class]);
    $this->router->post('/painel/servico', 'DashboardController@saveService', [CsrfMiddleware::class]);
    $this->router->post('/painel/servico-excluir', 'DashboardController@deleteService', [CsrfMiddleware::class]);
    $this->router->post('/painel/upload-logo', 'DashboardController@uploadLogo', [CsrfMiddleware::class]);
    $this->router->post('/painel/upload-galeria', 'DashboardController@uploadGallery', [CsrfMiddleware::class]);
    $this->router->post('/painel/horarios', 'DashboardController@saveHours', [CsrfMiddleware::class]);

    $this->router->get('/admin', 'AdminController@dashboard');
    $this->router->get('/admin/listings', 'AdminListingController@index');
    $this->router->post('/admin/listings/status', 'AdminListingController@setStatus', [CsrfMiddleware::class]);
    $this->router->post('/admin/listings/plan', 'AdminListingController@setPlan', [CsrfMiddleware::class]);
    $this->router->get('/admin/categorias', 'AdminCategoryController@index');
    $this->router->post('/admin/categorias/salvar', 'AdminCategoryController@save', [CsrfMiddleware::class]);
    $this->router->post('/admin/categorias/excluir', 'AdminCategoryController@delete', [CsrfMiddleware::class]);
    $this->router->get('/admin/avaliacoes', 'AdminReviewController@index');
    $this->router->post('/admin/avaliacoes/status', 'AdminReviewController@setStatus', [CsrfMiddleware::class]);
    $this->router->get('/admin/export/listings.csv', 'AdminExportController@listingsCsv');
    $this->router->get('/admin/export/reviews.csv', 'AdminExportController@reviewsCsv');
  }

  public function run(): void
  {
    $this->router->dispatch($_SERVER['REQUEST_METHOD'] ?? 'GET', $_SERVER['REQUEST_URI'] ?? '/');
  }
}
