<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Auth;
use App\Core\Helpers;
use App\Models\Listing;
use App\Models\Category;
use App\Models\Service;
use App\Models\Media;
use App\Models\Analytics;
use App\Services\GeoService;
use App\Services\UploadService;

final class DashboardController extends Controller
{
  private function ensureListing(): array
  {
    Auth::requireRole(['professional','company','admin']);
    $listing = Listing::findByUser((int)Auth::user()['id']);
    if (!$listing && (Auth::user()['role'] ?? '') !== 'admin') {
      http_response_code(403); echo "Seu perfil não foi criado."; exit;
    }
    return $listing ?: [];
  }

  public function index(): void
  {
    $listing = $this->ensureListing();
    $stats = $listing ? Analytics::statsForListing((int)$listing['id']) : [];
    $this->view('dashboard/index', compact('listing','stats'));
  }

  public function editProfile(): void
  {
    $listing = $this->ensureListing();
    $categories = Category::all();
    $selected = $listing ? array_column(Listing::categories((int)$listing['id']), 'id') : [];
    $services = $listing ? Listing::services((int)$listing['id']) : [];
    $media = $listing ? Listing::media((int)$listing['id']) : [];
    $logo = $listing ? Media::logoPath((int)$listing['id']) : null;
    $hours = $listing ? Listing::hours((int)$listing['id']) : [];
    $this->view('dashboard/profile', compact('listing','categories','selected','services','media','logo','hours'));
  }

  public function updateProfile(): void
  {
    $listing = $this->ensureListing();
    $id = (int)$listing['id'];

    $title = trim((string)($_POST['title'] ?? ''));
    if ($title === '') Helpers::redirect('/painel/perfil');

    $data = [
      'title' => $title,
      'bio' => trim((string)($_POST['bio'] ?? '')) ?: null,
      'phone' => trim((string)($_POST['phone'] ?? '')) ?: null,
      'whatsapp' => trim((string)($_POST['whatsapp'] ?? '')) ?: null,
      'email_public' => trim((string)($_POST['email_public'] ?? '')) ?: null,
      'website' => trim((string)($_POST['website'] ?? '')) ?: null,
      'budget_button_enabled' => !empty($_POST['budget_button_enabled']) ? 1 : 0,
      'street' => trim((string)($_POST['street'] ?? '')) ?: null,
      'number' => trim((string)($_POST['number'] ?? '')) ?: null,
      'district' => trim((string)($_POST['district'] ?? '')) ?: null,
      'city' => trim((string)($_POST['city'] ?? ($_ENV['DEFAULT_CITY'] ?? ''))),
      'state' => trim((string)($_POST['state'] ?? ($_ENV['DEFAULT_STATE'] ?? 'CE'))),
      'zipcode' => trim((string)($_POST['zipcode'] ?? '')) ?: null,
      'lat' => ($_POST['lat'] ?? '') !== '' ? (float)$_POST['lat'] : null,
      'lng' => ($_POST['lng'] ?? '') !== '' ? (float)$_POST['lng'] : null,
      'category_ids' => array_map('intval', (array)($_POST['category_ids'] ?? [])),
    ];

    Listing::updateProfile($id, $data);
    Helpers::redirect('/painel/perfil?ok=1');
  }

  public function geocodeAddress(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];
    $q = trim((string)($_POST['address_text'] ?? ''));
    $geo = (new GeoService())->geocode($q);
    if ($geo) Listing::setLatLng($id, (float)$geo['lat'], (float)$geo['lng']);
    Helpers::redirect('/painel/perfil?geo=' . ($geo ? 'ok' : 'fail'));
  }

  public function saveService(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];

    $sid = ($_POST['service_id'] ?? '') !== '' ? (int)$_POST['service_id'] : null;
    $name = trim((string)($_POST['name'] ?? ''));
    $desc = trim((string)($_POST['description'] ?? '')) ?: '';
    $price = ($_POST['price_from'] ?? '') !== '' ? (float)$_POST['price_from'] : null;
    $active = !empty($_POST['active']) ? 1 : 0;

    if ($name !== '') Service::upsert($sid, $id, $name, $desc, $price, $active);
    Helpers::redirect('/painel/perfil#servicos');
  }

  public function deleteService(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];
    $sid=(int)($_POST['service_id'] ?? 0);
    if ($sid>0) Service::delete($sid, $id);
    Helpers::redirect('/painel/perfil#servicos');
  }

  public function uploadLogo(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];
    $path = (new UploadService())->saveImage($_FILES['logo'] ?? [], 'logo');
    if ($path) Media::add($id, $path, 'logo');
    Helpers::redirect('/painel/perfil#midia');
  }

  public function uploadGallery(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];
    $files = $_FILES['gallery'] ?? null;
    if (!$files) Helpers::redirect('/painel/perfil#midia');

    // multi upload
    for ($i=0; $i < count($files['name']); $i++) {
      $f = [
        'name' => $files['name'][$i],
        'type' => $files['type'][$i],
        'tmp_name' => $files['tmp_name'][$i],
        'error' => $files['error'][$i],
        'size' => $files['size'][$i],
      ];
      $path = (new UploadService())->saveImage($f, 'gallery');
      if ($path) Media::add($id, $path, 'gallery');
    }
    Helpers::redirect('/painel/perfil#midia');
  }

  public function saveHours(): void
  {
    $listing = $this->ensureListing();
    $id=(int)$listing['id'];
    $pdo=\App\Core\DB::pdo();

    for ($d=0;$d<=6;$d++) {
      $open = $_POST['open'][$d] ?? null;
      $close = $_POST['close'][$d] ?? null;
      $closed = !empty($_POST['closed'][$d]) ? 1 : 0;

      $st=$pdo->prepare("INSERT INTO listing_hours (listing_id,weekday,open_time,close_time,is_closed)
        VALUES (?,?,?,?,?) ON DUPLICATE KEY UPDATE open_time=VALUES(open_time), close_time=VALUES(close_time), is_closed=VALUES(is_closed)");
      $st->execute([$id,$d,$open ?: null,$close ?: null,$closed]);
    }
    Helpers::redirect('/painel/perfil#horarios');
  }
}
