<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Auth;
use App\Core\Helpers;
use App\Core\RateLimiter;
use App\Models\User;
use App\Models\Listing;
use App\Services\MailService;

final class AuthController extends Controller
{
  public function loginForm(): void { $this->view('auth/login'); }

  public function login(): void
  {
    $email = trim((string)($_POST['email'] ?? ''));
    $pass  = (string)($_POST['password'] ?? '');
    $key = 'login:' . ($_SERVER['REMOTE_ADDR'] ?? 'ip') . ':' . $email;

    if (!RateLimiter::hit($key, 8, 15*60)) {
      http_response_code(429);
      $this->view('auth/login', ['error' => 'Muitas tentativas. Tente novamente em alguns minutos.']);
      return;
    }

    $user = User::findByEmail($email);
    if (!$user || !password_verify($pass, $user['password_hash'])) {
      $this->view('auth/login', ['error' => 'Credenciais inválidas.']);
      return;
    }
    if ($user['status'] === 'banned') {
      $this->view('auth/login', ['error' => 'Conta suspensa.']);
      return;
    }
    if ($user['status'] === 'pending') {
      $this->view('auth/login', ['error' => 'Confirme seu e-mail antes de entrar.']);
      return;
    }

    Auth::login($user);
    Helpers::redirect($user['role'] === 'admin' ? '/admin' : '/painel');
  }

  public function logout(): void
  {
    Auth::logout();
    Helpers::redirect('/');
  }

  public function registerForm(): void { $this->view('auth/register'); }

  public function register(): void
  {
    $name = trim((string)($_POST['name'] ?? ''));
    $email = trim((string)($_POST['email'] ?? ''));
    $pass = (string)($_POST['password'] ?? '');
    $role = in_array(($_POST['role'] ?? 'user'), ['user','professional','company'], true) ? $_POST['role'] : 'user';

    if ($name === '' || !filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($pass) < 8) {
      $this->view('auth/register', ['error' => 'Preencha corretamente. Senha mínima: 8 caracteres.']);
      return;
    }

    if (User::findByEmail($email)) {
      $this->view('auth/register', ['error' => 'E-mail já cadastrado.']);
      return;
    }

    $hash = password_hash($pass, PASSWORD_DEFAULT);
    $verifyToken = bin2hex(random_bytes(24));
    $verifyHash = hash('sha256', $verifyToken);

    $id = User::create($name, $email, $hash, $role, $verifyHash);

    // criar listing básico para professional/company
    if ($role === 'professional' || $role === 'company') {
      $type = $role === 'company' ? 'company' : 'professional';
      $title = $name;
      $slug = Helpers::slug($title . '-' . ($_ENV['DEFAULT_CITY'] ?? 'cidade') . '-' . $id);
      Listing::createForUser($id, $type, $title, $slug, $_ENV['DEFAULT_CITY'] ?? 'Fortaleza', $_ENV['DEFAULT_STATE'] ?? 'CE');
    }

    $link = Helpers::url('/verificar-email?email=' . urlencode($email) . '&token=' . urlencode($verifyToken));
    (new MailService())->send($email, 'Confirme seu e-mail', "Olá {$name}!\n\nConfirme seu e-mail acessando:\n{$link}\n\nSe não foi você, ignore.");

    $this->view('auth/register', ['success' => 'Cadastro realizado! Verifique seu e-mail (veja storage/logs/mail.log no modo log).']);
  }

  public function verifyEmail(): void
  {
    $email = (string)($_GET['email'] ?? '');
    $token = (string)($_GET['token'] ?? '');
    if (!filter_var($email, FILTER_VALIDATE_EMAIL) || $token === '') {
      $this->view('auth/verify', ['error' => 'Link inválido.']);
      return;
    }
    $ok = User::verifyEmail($email, hash('sha256', $token));
    $this->view('auth/verify', ['success' => $ok ? 'E-mail confirmado! Você já pode entrar.' : 'Link expirado ou inválido.']);
  }

  public function forgotForm(): void { $this->view('auth/forgot'); }

  public function forgot(): void
  {
    $email = trim((string)($_POST['email'] ?? ''));
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $this->view('auth/forgot', ['error' => 'Informe um e-mail válido.']);
      return;
    }

    $user = User::findByEmail($email);
    // sempre responder genérico
    if ($user) {
      $token = bin2hex(random_bytes(24));
      $hash = hash('sha256', $token);
      $exp = date('Y-m-d H:i:s', time() + 60*60);
      User::setResetToken((int)$user['id'], $hash, $exp);
      $link = Helpers::url('/resetar-senha?email=' . urlencode($email) . '&token=' . urlencode($token));
      (new MailService())->send($email, 'Recuperação de senha', "Acesse para redefinir (1h):\n{$link}");
    }
    $this->view('auth/forgot', ['success' => 'Se o e-mail existir, enviaremos um link de recuperação.']);
  }

  public function resetForm(): void
  {
    $this->view('auth/reset', [
      'email' => (string)($_GET['email'] ?? ''),
      'token' => (string)($_GET['token'] ?? '')
    ]);
  }

  public function reset(): void
  {
    $email = (string)($_POST['email'] ?? '');
    $token = (string)($_POST['token'] ?? '');
    $pass  = (string)($_POST['password'] ?? '');

    if (!filter_var($email, FILTER_VALIDATE_EMAIL) || $token === '' || strlen($pass) < 8) {
      $this->view('auth/reset', ['error' => 'Dados inválidos.', 'email'=>$email, 'token'=>$token]);
      return;
    }

    $u = User::findByReset($email, hash('sha256', $token));
    if (!$u) {
      $this->view('auth/reset', ['error' => 'Link inválido/expirado.', 'email'=>$email, 'token'=>$token]);
      return;
    }
    User::updatePassword((int)$u['id'], password_hash($pass, PASSWORD_DEFAULT));
    $this->view('auth/reset', ['success' => 'Senha atualizada. Você já pode entrar.']);
  }
}
